defmodule Jeu do
  @moduledoc """
  Point d'entrée du jeu de cartes.

  Ce module doit être appelé :

  - Pour créer une partie de jeu de cartes ;
  - Pour connaître la partie (PID) correspondante à un identifiant ;
  - Pour connaître la liste des parties en cours sur cette instance ou le cluster ;
  - Pour ajouter un joueur et jouer à une partie.

  Il est donc complètement découplé du système d'interface et peut être
  utilisé par une autre application, quelque soit l'interface implémentée.
  """

  use GenServer

  @doc """
  Lance le point d'entrée. Un seul point d'entrée doit être créé par instance.
  """
  @spec start_link(GenServer.options()) :: GenServer.on_start()
  def start_link(options \\ []) do
    options = Keyword.put(options, :name, :entrée)
    GenServer.start_link(__MODULE__, nil, options)
  end

  @doc """
  Vérifie qu'au moins une instance est capable de conserver les jeux de carte.
  Si le groupe `"cartes"`  est vide, aucune instance exécutant
  l'application `carte` n'a pu être trouvée.
  """
  def peut_communiquer?() do
    length(points_entrée()) != 0
  end

  @doc """
  Crée une nouvelle partie de jeu de cartes.

  On doit préciser en paramètre le module de règles (comme `Jeu.Huit` par exemple).
  L'identifiant de la partie est automatiquement recherché, ainsi que
  l'instance exécutant l'application `"carte"` la moins sollicitée.
  Celle-ci sera en charge de créer la partie de jeu de cartes. L'identifiant de
  la partie (un nombre) et l'identifiant du processus (un PID) seront
  retournés dans un tuple.
  """
  @spec créer_partie(module()) :: {integer(), pid()}
  def créer_partie(règles) do
    léger =
      Enum.min_by(points_entrée(), fn processus ->
        Jeu.nombre_parties(processus)
      end)

    GenServer.call(léger, {:créer_partie, règles})
  end

  @doc """
  Retourne le nombre de parties de jeu de cartes de l'instance du processus.
  """
  @spec nombre_parties(pid()) :: integer()
  def nombre_parties(processus) do
    GenServer.call(processus, :nombre_parties)
  end

  @doc """
  Retourne, si trouvé, le processus (PID) d'une partie.
  On doit préciser en paramètre l'identifiant de la partie recherchée.
  La recherche s'effectue sur toutes les instances exécutant l'application "carte".
  Si la recherche échoue, retourne `nil`.
  """
  @spec trouver_partie(integer()) :: pid() | nil
  def trouver_partie(identifiant) do
    Enum.find_value(points_entrée(), fn entrée ->
      GenServer.call(entrée, {:trouver_partie, identifiant})
    end)
  end

  @doc """
  Retourne un map (identifiant => {PID, titre}) de toutes les parties, peu importe leur instance.
  """
  @spec lister_parties() :: %{integer() => pid()}
  def lister_parties do
    parties =
      for entrée <- points_entrée() do
        GenServer.call(entrée, :lister_parties)
      end

    # Compresse cette liste des maps en un seul map.
    Enum.reduce(parties, &Map.merge/2)
  end

  @doc """
  Ajoute un joueur dans une partie, notifie le processus appelant.

  Cette fonction ajoute un joueur avec le nom donné dans la partie correspondante.
  Cette opération ne tient pas compte de l'emplacement de la partie,
  qu'elle soit sur cette instance ou une autre instance du cluster.
  Le processus appelant (`self/0`) sera notifié quand une modification
  apportée à cette partie se produira.

  Paramètres à renseigner :

  - `id` : l'identifiant de la partie (un nombre entier) ;
  - `processus` : le processus contenant la partie (un PID) ;
  - `nom` : le nom du joueur à ajouter (une chaîne de caractères).
  """
  @spec ajouter_joueur(integer(), pid(), String.t()) :: {:ok, integer()} | :invalide
  def ajouter_joueur(id, processus, nom) do
    entrée = List.first(points_entrée())
    résultat = GenServer.call(entrée, {:ajouter_joueur, id, processus, nom})

    case résultat do
      {joueur, _} ->
        :pg.join({:joueur, id}, self())
        notifier_joueur(self())
      {:ok, joueur}

    :invalide ->
      :invalide
    end
  end

  @doc """
  Joue un coup dans une partie, notifie les joueurs inscrits.

  Cette fonction permet au joueur dont l'identifiant est précisé
  en paramètre de jouer un coup, également précisé en paramètre.
  Cette opération ne tient pas compte de l'emplacement de la partie,
  qu'elle soit sur cette instance ou une autre instance du cluster.
  Les joueurs inscrits à cette partie (un groupe de processus par partie)
  seront notifiés.

  Paramètres à renseigner :

  - `id` : l'identifiant de la partie (un nombre entier) ;
  - `processus` : le processus contenant la partie (un PID) ;
  - `joueur` : le joueur sous la forme d'un identifiant (un nombre entier) ;
  - `coup` : le coup à jouer (un atome ou tuple).
  """
  @spec jouer(integer(), pid(), integer(), atom() | tuple()) :: :ok | :invalide
  def jouer(id, processus, joueur, coup) do
    entrée = List.first(points_entrée())
    résultat = GenServer.call(entrée, {:jouer, id, processus, joueur, coup})
    résultat != :invalide && :ok || :invalide
  end

  @impl true
  def init(_) do
    :pg.join("cartes", self())
    {:ok, {0, %{}}}
  end

  @impl true
  def handle_call(:nombre_parties, _from, {identifiant_max, parties}) do
    {:reply, map_size(parties), {identifiant_max, parties}}
  end

  @impl true
  def handle_call({:créer_partie, règles}, _from, {identifiant_max, parties}) do
    identifiant = :erlang.phash2({node(), identifiant_max})

    {:ok, nouvelle_partie} =
      DynamicSupervisor.start_child(
        Partie.Superviseur,
        {Partie, {identifiant, règles}}
      )

    parties = Map.put(parties, identifiant, {nouvelle_partie, règles.titre()})
    {:reply, {identifiant, nouvelle_partie}, {identifiant_max + 1, parties}}
  end

  @impl true
  def handle_call({:trouver_partie, identifiant}, _from, {identifiant_max, parties}) do
    {partie, _} = Map.get(parties, identifiant, {nil, nil})
    {:reply, partie, {identifiant_max, parties}}
  end

  @impl true
  def handle_call(:lister_parties, _from, {identifiant_max, parties}) do
    {:reply, parties, {identifiant_max, parties}}
  end

  @impl true
  def handle_call({:ajouter_joueur, id, processus, nom}, _from, {identifiant_max, parties}) do
    résultat = Partie.ajouter_joueur(processus, nom)

    if résultat != :invalide do
      notifier_joueurs(id)
    end

    {:reply, résultat, {identifiant_max, parties}}
  end

  @impl true
  def handle_call({:jouer, id, processus, joueur, coup}, _from, {identifiant_max, parties}) do
    résultat = Partie.jouer(processus, joueur, coup)

    if résultat != :invalide do
      notifier_joueurs(id)
    end

    {:reply, résultat, {identifiant_max, parties}}
  end

  @spec points_entrée() :: [pid()]
  defp points_entrée() do
    :pg.get_members("cartes")
  end

  @spec notifier_joueurs(integer()) :: :ok
  defp notifier_joueurs(id) do
    Enum.each(:pg.get_members({:joueur, id}), fn joueur ->
      notifier_joueur(joueur)
    end)
  end

  @spec notifier_joueur(pid()) :: :actualiser
  defp notifier_joueur(joueur) do
    send(joueur, :actualiser)
  end
end
